<?php
// guardar_loteria.php — Crear rifa (JSON)
declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

/* ===== Siempre responder JSON ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: POST, OPTIONS');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Chequeo de sesión ===== */
if (!isset($_SESSION['idusuario'])) {
  echo json_encode(['success'=>false,'message'=>'Sesión no iniciada.']); exit;
}
$id_usuario  = (int)$_SESSION['idusuario'];
$sucursal_id = $_SESSION['sucursal'] ?? null;
$my_ip       = $_SERVER['REMOTE_ADDR'] ?? '';

/* ===== Conexión ===== */
require __DIR__ . '/conex.php';
$cn = $conn ?? ($conex ?? null);
if (!$cn instanceof mysqli) {
  echo json_encode(['success'=>false,'message'=>'No se pudo establecer conexión con la base de datos.']); exit;
}
if (function_exists('mysqli_report')) {
  mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
}
mysqli_set_charset($cn, 'utf8mb4');

/* ===== Helpers ===== */
function post_s(string $k): string {
  return isset($_POST[$k]) ? trim((string)$_POST[$k]) : '';
}
function to_time_hms(string $hhmm): string {
  // Normaliza "HH:MM" / "HH:MM:SS" / "HHMM" a "HH:MM:SS"
  $hhmm = trim($hhmm);
  if ($hhmm === '') return '';
  if (preg_match('/^\d{4}$/', $hhmm)) { // "0600"
    $hh = substr($hhmm, 0, 2); $mm = substr($hhmm, 2, 2);
    return sprintf('%02d:%02d:00', (int)$hh, (int)$mm);
  }
  if (preg_match('/^\d{2}:\d{2}$/', $hhmm)) {
    return $hhmm . ':00';
  }
  if (preg_match('/^\d{2}:\d{2}:\d{2}$/', $hhmm)) {
    return $hhmm;
  }
  return ''; // formato inválido
}

/* ===== Entradas ===== */
$fecha       = post_s('fecha');       // YYYY-MM-DD
$dia         = post_s('dia');         // Lunes/Martes/...
$loteria_in  = post_s('loteria');     // texto o código tal cual llega
$hora_inicio = to_time_hms(post_s('hora_inicio'));
$hora_fin    = to_time_hms(post_s('hora_fin'));

$activar   = 0; // por defecto desactivada (cámbialo a 1 si quieres activarlas al crear)
$condicion = 1;

/* ===== Validaciones ===== */
if ($fecha === '' || $dia === '' || $loteria_in === '' || $hora_inicio === '' || $hora_fin === '') {
  echo json_encode(['success'=>false,'message'=>'Todos los campos son obligatorios.']); exit;
}
if (strtotime($hora_inicio) === false || strtotime($hora_fin) === false) {
  echo json_encode(['success'=>false,'message'=>'Formato de hora inválido. Use HH:MM.']); exit;
}
if ($hora_inicio >= $hora_fin) {
  echo json_encode(['success'=>false,'message'=>'La hora de inicio debe ser menor que la de fin.']); exit;
}
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $fecha)) {
  echo json_encode(['success'=>false,'message'=>'Fecha inválida (use YYYY-MM-DD).']); exit;
}

/* ===== Mapeo de país según nombre de lotería ===== */
$loteria = trim($loteria_in);
$loteriaUC = mb_strtoupper($loteria, 'UTF-8');

$listaARM = [
  'CON MATUTINA',  
];
$listaARV = [  
  'CON VESPERTINA',
];
$listaEC = [
  'CON LTRA DOM',
  'CON LTRA LUN',
  'CON LTT MAR',
  'CON LTRA MIE',
  'CON LTT JUE',
  'CON LTRA VIE',
  'CON LTT SAB',
];

if (in_array($loteriaUC, $listaARM, true)) {
  $id_pais = 2; // Argentina
  $id_premio= 2;
} elseif (in_array($loteriaUC, $listaARV, true)) {
  $id_pais = 2; // Ecuador
  $id_premio= 3;
}elseif (in_array($loteriaUC, $listaEC, true)) {
  $id_pais = 1; // Ecuador
  $id_premio= 1;
} else {
  $id_pais = 0; // Desconocido/otro (ajusta a NULL si tu columna lo permite)
}

try {
  // Columnas esperadas en tbr_loteria:
  // (fecha DATE, dia VARCHAR, loteria VARCHAR, hora_inicio TIME, hora_fin TIME, activar TINYINT, id_pais INT, condicion TINYINT)
  $sql = "INSERT INTO tbr_loteria
            (fecha, dia, loteria, hora_inicio, hora_fin, activar, id_pais, id_premio, condicion)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
  $st = $cn->prepare($sql);

  // 5 strings (fecha, dia, loteria, hora_inicio, hora_fin) + 3 enteros (activar, id_pais, condicion)
  $st->bind_param(
    'sssssiiii',
    $fecha,        // s
    $dia,          // s
    $loteria,      // s (en el mismo casing que llegó)
    $hora_inicio,  // s
    $hora_fin,     // s
    $activar,      // i
    $id_pais,      // i
    $id_premio,   // i
    $condicion     // i
  );

  $st->execute();
  $insert_id = $cn->insert_id;
  $st->close();

  echo json_encode([
    'success'    => true,
    'message'    => 'Rifa registrada exitosamente.',
    'idloteria'  => $insert_id,
    'id_pais'    => $id_pais,
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  echo json_encode([
    'success' => false,
    'message' => 'Error al insertar la rifa.',
    'error'   => $e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
}
