<?php
// guardar_pago.php — Pago idempotente con mensaje + fecha/hora si ya está pagado
declare(strict_types=1);
session_start();
require_once __DIR__ . '/conex.php';
header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('America/Guayaquil');

/* ===== Validaciones ===== */
if (!isset($_SESSION['idusuario'])) {
  echo json_encode(['success'=>false,'message'=>'Sesión no iniciada']); exit;
}
if (!isset($_POST['idrifa']) || !is_numeric($_POST['idrifa'])) {
  echo json_encode(['success'=>false,'message'=>'Parámetro idrifa inválido']); exit;
}
if (!isset($conn) || !$conn) {
  echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit;
}

$id_usuario = (int) $_SESSION['idusuario'];
$idrifa     = (int) $_POST['idrifa'];
$ip         = $_SERVER['REMOTE_ADDR'] ?? '';
$condicion  = 1;

$fecha_actual = date('Y-m-d'); // formato DATE correcto
$hora_actual  = date('H:i:s'); // H:i:s (i = minutos)

mysqli_set_charset($conn, 'utf8mb4');

try {
  if (function_exists('mysqli_report')) {
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
  }

  // 1) Intento idempotente: solo cambia 2 -> 3
  $sqlUpd = "UPDATE tbr_ticket SET estado = 3 WHERE idrifa = ? AND estado = 2";
  $stmtUpd = mysqli_prepare($conn, $sqlUpd);
  mysqli_stmt_bind_param($stmtUpd, "i", $idrifa);
  mysqli_stmt_execute($stmtUpd);
  $affected = mysqli_stmt_affected_rows($stmtUpd);
  mysqli_stmt_close($stmtUpd);

  if ($affected === 1) {
    // 2) Registramos pago una sola vez
    $sqlIns = "INSERT INTO tbr_pago (fecha, hora, ip, id_usuario, id_ticket, condicion)
               VALUES (?, ?, ?, ?, ?, ?)";
    $stmtIns = mysqli_prepare($conn, $sqlIns);
    mysqli_stmt_bind_param($stmtIns, "sssiii", $fecha_actual, $hora_actual, $ip, $id_usuario, $idrifa, $condicion);
    mysqli_stmt_execute($stmtIns);
    mysqli_stmt_close($stmtIns);

    echo json_encode([
      'success'     => true,
      'message'     => 'Pago aplicado',
      'idrifa'      => $idrifa,
      'fecha_pago'  => $fecha_actual,
      'hora_pago'   => $hora_actual
    ]);
    exit;
  }

  // 3) Si no afectó filas, verificar estado actual para responder claro
  $sqlSel = "SELECT estado FROM tbr_ticket WHERE idrifa = ? LIMIT 1";
  $stmtSel = mysqli_prepare($conn, $sqlSel);
  mysqli_stmt_bind_param($stmtSel, "i", $idrifa);
  mysqli_stmt_execute($stmtSel);
  $resSel = mysqli_stmt_get_result($stmtSel);
  $rowSel = mysqli_fetch_assoc($resSel);
  mysqli_stmt_close($stmtSel);

  if (!$rowSel) {
    echo json_encode(['success'=>false,'message'=>'Ticket no encontrado']); exit;
  }

  $estado = (int) $rowSel['estado'];
  if ($estado === 3) {
    // Buscar fecha/hora del último pago
    $sqlPago = "SELECT fecha, hora
                FROM tbr_pago
                WHERE id_ticket = ? AND condicion = 1
                ORDER BY fecha DESC, hora DESC
                LIMIT 1";
    $stmtPago = mysqli_prepare($conn, $sqlPago);
    mysqli_stmt_bind_param($stmtPago, "i", $idrifa);
    mysqli_stmt_execute($stmtPago);
    $resPago = mysqli_stmt_get_result($stmtPago);
    $rowPago = mysqli_fetch_assoc($resPago);
    mysqli_stmt_close($stmtPago);

    echo json_encode([
      'success'    => false,
      'code'       => 'already_paid',
      'message'    => 'ticket ya se encuentra pagado',  // ← texto exacto solicitado
      'idrifa'     => $idrifa,
      'fecha_pago' => $rowPago['fecha'] ?? null,
      'hora_pago'  => $rowPago['hora'] ?? null
    ]);
    exit;
  }

  echo json_encode([
    'success' => false,
    'code'    => 'not_payable',
    'message' => 'Ticket no está pendiente de pago'
  ]);
} catch (Throwable $e) {
  echo json_encode([
    'success' => false,
    'message' => 'Error al procesar el pago',
    'error'   => $e->getMessage()
  ]);
}
