<?php
// guardar_ticket3.php  Inserta ticket (3 cifras)
declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: POST, OPTIONS');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

try {
    // ==== Validar sesin ====
    if (!isset($_SESSION['idusuario'], $_SESSION['sucursal'])) {
        echo json_encode([
            'success' => false,
            'message' => 'Sesin no iniciada'
        ]);
        exit;
    }

    $id_usuario  = (int) $_SESSION['idusuario'];
    $id_sucursal = (int) $_SESSION['sucursal'];

    // ==== Conexin BD ====
    require __DIR__ . '/conex.php';
    $cn = $conn ?? ($conex ?? null);
    if (!$cn instanceof mysqli) {
        echo json_encode([
            'success' => false,
            'message' => 'Sin conexin BD'
        ]);
        exit;
    }

    if (function_exists('mysqli_report')) {
        mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
    }
    mysqli_set_charset($cn, 'utf8mb4');
    @mysqli_query($cn, "SET time_zone = '-05:00'");

    // Helper para POST
    $post = fn(string $k): string => isset($_POST[$k]) ? trim((string) $_POST[$k]) : '';

    // ==== Datos de entrada ====
    $date2     = $post('date2');
    $idloteria = (int) ($_POST['loterias'] ?? 0);
    $valor     = (float) ($_POST['valor'] ?? 0);

    // Normalizar nmero a 3 cifras
    $rawNum = preg_replace('/\D/', '', $post('number')); // Solo dgitos
    $number = str_pad(substr($rawNum, -3), 3, '0', STR_PAD_LEFT);

    // id_cifras: 1 = 2 cifras, 2 = 3 cifras, 3 = 4 cifras
    $id_cifras = 2; // 3 cifras

    $hoy = date('Y-m-d');

    // ==== Validaciones bsicas ====
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date2)) {
        throw new Exception('Fecha invlida (YYYY-MM-DD).');
    }
    if ($date2 < $hoy) {
        throw new Exception('La fecha no puede ser menor a hoy.');
    }
    if ($idloteria <= 0) {
        throw new Exception('Seleccione una lotera.');
    }
    if (!preg_match('/^\d{3}$/', $number)) {
        throw new Exception('Nmero invlido (000-999).');
    }
    if ($valor < 0.50) {
        throw new Exception('El valor debe ser  0.50.');
    }

    // ==== Validar hora de cierre de la lotera ====
    $horaHi = (int) date('Hi'); // HHMM actual
    $HCAST  = "CAST(LPAD(LEFT(REPLACE(hora_fin, ':',''), 4), 4, '0') AS UNSIGNED)";

    $st = $cn->prepare("SELECT $HCAST AS finHHMM, activar, condicion FROM tbr_loteria WHERE idloteria = ?");
    $st->bind_param('i', $idloteria);
    $st->execute();
    $lot = $st->get_result()->fetch_assoc();
    $st->close();

    if (
        !$lot ||
        (int) $lot['activar'] !== 1 ||
        (int) $lot['condicion'] !== 1
    ) {
        throw new Exception('Lotera inactiva o inexistente.');
    }

    if ($date2 === $hoy && (int) $lot['finHHMM'] < $horaHi) {
        throw new Exception('Fuera de la hora de juego permitida.');
    }

    // ==== CUPOS ====
    // Cupo global por cifra (tabla tbr_cifras)
    $cupo_suc = 0.0;
    $sqlCupo = "SELECT cupo_cifras FROM tbr_cifras WHERE idcifras = ?";
    if ($stmt = $cn->prepare($sqlCupo)) {
        $stmt->bind_param("i", $id_cifras);
        $stmt->execute();
        $res = $stmt->get_result();
        if ($row = $res->fetch_assoc()) {
            $cupo_suc = (float) $row['cupo_cifras'];
        }
        $stmt->close();
    }

    // Cupo adicional por usuario (extra_cupo) si cupo_activo = 1
    $st = $cn->prepare("
        SELECT CASE WHEN cupo_activo = 1 THEN COALESCE(extra_cupo, 0) ELSE 0 END AS extra
        FROM tbr_usuario
        WHERE idusuario = ?
    ");
    $st->bind_param('i', $id_usuario);
    $st->execute();
    $extra = (float) ($st->get_result()->fetch_assoc()['extra'] ?? 0);
    $st->close();

    // Suma ya jugada PARA ESA LOTER01A, esa fecha, nmero, sucursal y 3 cifras
    $sql = "
        SELECT IFNULL(SUM(t.valor), 0) AS suma
        FROM tbr_ticket t
        INNER JOIN tbr_usuario u ON u.idusuario = t.id_usuario
        WHERE t.fecha       = ?
          AND t.condicion   = 1
          AND t.id_cifras   = ?
          AND t.numero      = ?
          AND u.id_sucursal = ?
          AND t.id_loteria  = ?
    ";
    $st = $cn->prepare($sql);
    // s = fecha, i = id_cifras, s = numero, i = id_sucursal, i = idloteria
    $st->bind_param('sisii', $date2, $id_cifras, $number, $id_sucursal, $idloteria);
    $st->execute();
    $suma = (float) ($st->get_result()->fetch_assoc()['suma'] ?? 0);
    $st->close();

    // Cupo total = cupo global + extra
    $cupo_total      = $cupo_suc + $extra;
    $restante_total  = $cupo_total - $suma;

    if (($suma + $valor) > $cupo_total) {
        echo json_encode([
            'success'        => false,
            'error'          => 'Cupo_no_disponible',
            'mensaje'        => "No puedes superar el cupo mximo de $cupo_total para esta lotera.",
            'cupo_restante'  => number_format(max($restante_total, 0), 2, '.', '')
        ]);
        exit;
    }

    // Distribucin entre cupo normal (cupo_suc) y extra (extra_cupo)
    $restante_sucursal = max($cupo_suc - $suma, 0);
    $tope1 = min($valor, $restante_sucursal);    // Lo que va al cupo normal
    $tope2 = max($valor - $tope1, 0);           // Lo que usara del extra

    if ($tope2 > $extra) {
        throw new Exception("No puedes superar el cupo adicional de $extra.");
    }

    // ==== Insertar ticket ====
    $fecha_imp = date('Y-m-d');
    $hora_imp  = date('H:i:s');
    $codigo    = substr(str_shuffle(str_repeat('0123456789', 8)), 0, 8);
    $estado    = 1;
    $cond      = 1;

    $sql = "INSERT INTO tbr_ticket
        (fecha_impreso, hora_impreso, fecha, id_loteria, numero, valor,
         id_usuario, codigo, estado, tope1, tope2, id_cifras, condicion)
        VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?)";

    $st = $cn->prepare($sql);
    $st->bind_param(
        'sssisdisiddii',
        $fecha_imp,
        $hora_imp,
        $date2,
        $idloteria,
        $number,
        $valor,
        $id_usuario,
        $codigo,
        $estado,
        $tope1,
        $tope2,
        $id_cifras,
        $cond
    );
    $st->execute();
    $id = $cn->insert_id;
    $st->close();

    echo json_encode([
        'success'        => true,
        'id'             => $id,
        'codigo'         => $codigo,
        'numero'         => $number,
        'valor'          => number_format($valor, 2, '.', ''),
        'cupo_restante'  => number_format(max($restante_total - $valor, 0), 2, '.', '')
    ]);
} catch (Throwable $e) {
    echo json_encode([
        'success' => false,
        'message' => 'No se pudo guardar',
        'error'   => $e->getMessage()
    ]);
}
