<?php
// ticket_list4c.php — Carrito (4 cifras) GLOBAL del usuario (sin filtros de fecha/lotería)
declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/conex.php';
if (!isset($conn) || !$conn instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'data'=>[],'message'=>'Sin conexión a BD']); exit;
}

if (function_exists('mysqli_report')) {
  mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
}
mysqli_set_charset($conn, 'utf8mb4');

// Seguridad básica
if (!isset($_SESSION['idusuario'])) {
  http_response_code(401);
  echo json_encode(['success'=>false,'data'=>[],'message'=>'Sesión expirada']); exit;
}

$idusuario = (int)$_SESSION['idusuario'];

try {
  /**
   * Estructura usada:
   * - tbr_ticket: idrifa (PK), id_usuario, fecha (DATE), id_loteria, numero, valor, id_cifras, estado, condicion
   * - tbr_loteria: idloteria (PK), loteria (nombre)
   *
   * Carrito GLOBAL = tickets del usuario, pendientes (estado=1), válidos (condicion=1), de 2 cifras (id_cifras=1).
   * ❗ Sin filtro por fecha ni por id_loteria.
   */
  $sql = "
    SELECT
      t.idrifa                         AS id,
      COALESCE(l.loteria, '')          AS loteria,
      DATE_FORMAT(t.fecha, '%Y-%m-%d') AS fecha,
      LPAD(t.numero, 4, '0')           AS numero,
      t.valor
    FROM tbr_ticket AS t
    LEFT JOIN tbr_loteria AS l ON l.idloteria = t.id_loteria
    WHERE
      t.condicion = 1
      AND t.id_cifras = 3
      AND t.estado = 1             -- pendiente / cargado sin imprimir
      AND t.id_usuario = ?
    ORDER BY t.fecha ASC, t.idrifa ASC
  ";

  $stmt = $conn->prepare($sql);
  $stmt->bind_param('i', $idusuario);
  $stmt->execute();
  $res = $stmt->get_result();

  $data  = [];
  $total = 0.0;
  while ($row = $res->fetch_assoc()) {
    $row['valor'] = (float)$row['valor'];
    $total += $row['valor'];
    $data[] = $row;
  }
  $stmt->close();

  echo json_encode([
    'success' => true,
    'data'    => $data,
    'total'   => round($total, 2)
  ]);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success' => false,
    'data'    => [],
    'message' => $e->getMessage()
  ]);
}
