<?php
// tickets_eliminados_list.php (mismo directorio que tu index actual)
declare(strict_types=1);
session_start();
header('Content-Type: application/json; charset=utf-8');

// Opcional: evita que warnings se impriman y rompan el JSON
ini_set('display_errors', '0');
ini_set('log_errors', '1');

if (!isset($_SESSION['idusuario'])) {
  echo json_encode(['data' => [], 'error' => 'Sesión no válida'], JSON_UNESCAPED_UNICODE);
  exit;
}

// ⬇️ Si este .php está en el MISMO directorio que conex.php:
require_once 'conex.php';

// ⬇️ Si conex.php está un nivel arriba, usa: require_once __DIR__ . '/../conex.php';

if (!isset($conn) || !($conn instanceof mysqli)) {
  echo json_encode(['data' => [], 'error' => 'Sin conexión a BD'], JSON_UNESCAPED_UNICODE);
  exit;
}

mysqli_set_charset($conn, 'utf8mb4');
@mysqli_query($conn, "SET time_zone = '-05:00'"); // Ecuador

// Filtros GET
$fecha_i = $_GET['fecha_i'] ?? '';
$fecha_f = $_GET['fecha_f'] ?? '';
$id_suc  = isset($_GET['sucursal']) && $_GET['sucursal'] !== '' ? (int)$_GET['sucursal'] : 0;
$id_user = isset($_GET['usuario'])  && $_GET['usuario']  !== '' ? (int)$_GET['usuario']  : 0;

$pat = '/^\d{4}-\d{2}-\d{2}$/';
if ($fecha_i !== '' && !preg_match($pat, $fecha_i)) $fecha_i = '';
if ($fecha_f !== '' && !preg_match($pat, $fecha_f)) $fecha_f = '';

if ($fecha_i === '' || $fecha_f === '') {
  $hoy   = new DateTime('now', new DateTimeZone('America/Guayaquil'));
  $desde = (clone $hoy)->modify('-15 days');
  $fecha_i = $desde->format('Y-m-d');
  $fecha_f = $hoy->format('Y-m-d');
}

// Fecha impreso en hora de Ecuador (desde UTC) con fallback -5h
$impreso_ec = "IFNULL(
  CONVERT_TZ(TIMESTAMP(t.fecha_impreso, t.hora_impreso), 'UTC','America/Guayaquil'),
  TIMESTAMP(t.fecha_impreso, t.hora_impreso) - INTERVAL 5 HOUR
)";

// WHERE base y rango
$where = ["t.condicion = 0"];
if ($id_suc > 0)  $where[] = "s.idsucursal = $id_suc";
if ($id_user > 0) $where[] = "u.idusuario  = $id_user";
$where[] = "DATE($impreso_ec) BETWEEN '$fecha_i' AND '$fecha_f'";
$where_sql = 'WHERE ' . implode(' AND ', $where);

$sql = "
SELECT 
  c.cifras,
  s.sucursal,
  u.usuario,
  l.loteria,
  t.fecha_impreso,
  t.hora_impreso,
  $impreso_ec AS impreso_ecuador,
  t.numero,
  t.valor,
  t.codigo,
  t.condicion,
  t.fecha_hora_eliminado
FROM tbr_ticket AS t
JOIN tbr_usuario  AS u ON u.idusuario  = t.id_usuario
JOIN tbr_cifras   AS c ON c.idcifras   = t.id_cifras
JOIN tbr_loteria  AS l ON l.idloteria  = t.id_loteria
JOIN tbr_sucursal AS s ON s.idsucursal = u.id_sucursal
$where_sql
ORDER BY impreso_ecuador DESC, t.idrifa DESC
";

$res = mysqli_query($conn, $sql);
if (!$res) {
  echo json_encode(['data' => [], 'error' => 'Error SQL: ' . mysqli_error($conn)], JSON_UNESCAPED_UNICODE);
  exit;
}

// Convierte 'YYYY/MM/DD HH:MM:SS' o 'YYYY-MM-DD HH:MM:SS' (UTC) → America/Guayaquil
function parseUtcToGye(?string $txt): ?string {
  if (!$txt) return null;
  $txt = trim($txt);
  if ($txt === '') return null;

  $dt = null;
  if (preg_match('/^\d{4}\/\d{2}\/\d{2} \d{2}:\d{2}:\d{2}$/', $txt)) {
    $dt = DateTime::createFromFormat('Y/m/d H:i:s', $txt, new DateTimeZone('UTC'));
  } elseif (preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/', $txt)) {
    $dt = DateTime::createFromFormat('Y-m-d H:i:s', $txt, new DateTimeZone('UTC'));
  }
  if (!$dt) return null;
  $dt->setTimezone(new DateTimeZone('America/Guayaquil'));
  return $dt->format('Y-m-d H:i:s');
}

$data = [];
while ($row = mysqli_fetch_assoc($res)) {
  $cifras     = (int)$row['cifras'];
  $numero_fmt = str_pad((string)$row['numero'], max(1,$cifras), '0', STR_PAD_LEFT);
  $valor_fmt  = number_format((float)$row['valor'], 2, '.', ',');

  $fecha_impreso_ec = $row['impreso_ecuador'] ? date('Y-m-d H:i:s', strtotime($row['impreso_ecuador'])) : null;
  $fecha_elim_ec    = parseUtcToGye($row['fecha_hora_eliminado']);

  $cond = (string)$row['condicion'];
  $eliminado_label = ($cond === '0') ? 'Sí' : (($cond === '1') ? 'No' : 'Desconocido');

  $data[] = [
    'cifras'          => $row['cifras'],
    'sucursal'        => $row['sucursal'],
    'usuario'         => $row['usuario'],
    'fecha'           =>$row['fecha_impreso'].' '.$row['hora_impreso'],    
    'sorteo'          => $row['loteria'],
    'numero'          => $numero_fmt,
    'valor'           => $valor_fmt,
    'codigo'          => $row['codigo'],
    'eliminado'       => $eliminado_label,
    'fecha_eliminado' => $row['fecha_hora_eliminado']        // Eliminado Ecuador
  ];
}

echo json_encode(['data' => $data], JSON_UNESCAPED_UNICODE);
