<?php
// get_marca_stats.php — Stats de pagos por marca (por id o por nombre)
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

// Ajusta la ruta a conex.php si tu estructura es distinta
require_once __DIR__ . '/conex.php';

$DB = null;
if (isset($conex) && $conex instanceof mysqli)      { $DB = $conex; }
elseif (isset($conn) && $conn instanceof mysqli)    { $DB = $conn; }
if (!$DB) { echo json_encode(['success'=>false,'message'=>'Sin conexión a BD']); exit; }

@mysqli_set_charset($DB, 'utf8mb4');
@$DB->query("SET time_zone = '-05:00'");

$id      = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$nombre  = isset($_GET['nombre']) ? trim((string)$_GET['nombre']) : '';

if ($id <= 0 && $nombre === '') {
  echo json_encode(['success'=>false,'message'=>'Parámetros inválidos']); exit;
}

/*
Relaciones:
- tb_marca.id       → tb_publicidad.id_marca
- tb_publicidad.id  → tb_pago.id_publicidad
*/
if ($id > 0) {
  $sql = "
    SELECT m.id, m.plan, m.tiempo, m.valor,
           COALESCE(COUNT(p.id),0)  AS pagos_realizados,
           COALESCE(SUM(p.valor),0) AS total_pagado
    FROM tb_marca AS m
    LEFT JOIN tb_publicidad AS pub ON pub.id_marca = m.id
    LEFT JOIN tb_pago        AS p   ON p.id_publicidad = pub.id
    WHERE m.id = ?
    GROUP BY m.id, m.plan, m.tiempo, m.valor
    LIMIT 1";
  $stmt = $DB->prepare($sql);
  if (!$stmt) { echo json_encode(['success'=>false,'message'=>'Error de preparación']); exit; }
  $stmt->bind_param('i', $id);
} else {
  $sql = "
    SELECT m.id, m.plan, m.tiempo, m.valor,
           COALESCE(COUNT(p.id),0)  AS pagos_realizados,
           COALESCE(SUM(p.valor),0) AS total_pagado
    FROM tb_marca AS m
    LEFT JOIN tb_publicidad AS pub ON pub.id_marca = m.id
    LEFT JOIN tb_pago        AS p   ON p.id_publicidad = pub.id
    WHERE TRIM(LOWER(m.nombre_marca)) = TRIM(LOWER(?))
    GROUP BY m.id, m.plan, m.tiempo, m.valor
    LIMIT 1";
  $stmt = $DB->prepare($sql);
  if (!$stmt) { echo json_encode(['success'=>false,'message'=>'Error de preparación']); exit; }
  $stmt->bind_param('s', $nombre);
}

$stmt->execute();
$res = $stmt->get_result();
if (!$res || $res->num_rows === 0) {
  echo json_encode(['success'=>false,'message'=>'Marca no encontrada']); exit;
}

$row        = $res->fetch_assoc();
$tiempo     = (int)($row['tiempo'] ?? 0);
$realizados = (int)($row['pagos_realizados'] ?? 0);
$pendientes = max(0, $tiempo - $realizados);
$porcentaje = $tiempo > 0 ? round(($realizados / $tiempo) * 100, 2) : ($realizados > 0 ? 100.0 : 0.0);

echo json_encode([
  'success'=>true,
  'data'=>[
    'id'                => (int)$row['id'],
    'plan'              => (string)($row['plan'] ?? ''),
    'tiempo'            => $tiempo,
    'valor'             => isset($row['valor']) ? (float)$row['valor'] : 0.0,
    'pagos_realizados'  => $realizados,
    'pagos_pendientes'  => $pendientes,
    'pagos_esperados'   => $tiempo,
    'total_pagado'      => isset($row['total_pagado']) ? (float)$row['total_pagado'] : 0.0,
    'porcentaje_avance' => $porcentaje
  ]
], JSON_UNESCAPED_UNICODE | JSON_INVALID_UTF8_SUBSTITUTE);
