<?php
// subir_video.php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

// === SOLO EN DESARROLLO ===
// ini_set('display_errors', '1'); error_reporting(E_ALL);

require_once __DIR__ . '/conex.php'; // Debe definir $conex (mysqli)

function json_out(int $code, array $payload): void {
    http_response_code($code);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE);
    exit;
}

// Método
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
    json_out(405, ['success'=>false, 'message'=>'Método no permitido.']);
}

// Campos
$descripcion = trim((string)($_POST['descripcion'] ?? ''));
if ($descripcion === '') {
    json_out(400, ['success'=>false, 'message'=>'La descripción no puede estar vacía.']);
}

// Archivo
if (!isset($_FILES['video'])) {
    json_out(400, ['success'=>false, 'message'=>'No se recibió el archivo.']);
}

// Mapear errores comunes
$err = (int)($_FILES['video']['error'] ?? UPLOAD_ERR_NO_FILE);
if ($err !== UPLOAD_ERR_OK) {
    $map = [
        UPLOAD_ERR_INI_SIZE   => 'El archivo excede upload_max_filesize.',
        UPLOAD_ERR_FORM_SIZE  => 'El archivo excede MAX_FILE_SIZE del formulario.',
        UPLOAD_ERR_PARTIAL    => 'El archivo se subió parcialmente.',
        UPLOAD_ERR_NO_FILE    => 'No se subió ningún archivo.',
        UPLOAD_ERR_NO_TMP_DIR => 'Falta el directorio temporal.',
        UPLOAD_ERR_CANT_WRITE => 'No se pudo escribir el archivo.',
        UPLOAD_ERR_EXTENSION  => 'Una extensión de PHP detuvo la subida.',
    ];
    $msg = $map[$err] ?? 'Error desconocido al subir el archivo.';
    json_out(400, ['success'=>false, 'message'=>$msg]);
}

// Validar tamaño (<= 21 MB)
$tamanoMax = 21 * 1024 * 1024;
$size = (int)($_FILES['video']['size'] ?? 0);
if ($size <= 0) {
    json_out(400, ['success'=>false, 'message'=>'Archivo vacío o inválido.']);
}
if ($size > $tamanoMax) {
    json_out(413, ['success'=>false, 'message'=>'El archivo supera los 21 MB permitidos.']);
}

// Validar extensión declarada
$nombreOriginal = (string)($_FILES['video']['name'] ?? 'video.mp4');
$ext = strtolower(pathinfo($nombreOriginal, PATHINFO_EXTENSION));
if ($ext !== 'mp4') {
    json_out(400, ['success'=>false, 'message'=>'Solo se permiten archivos MP4.']);
}

// Validar MIME real (finfo)
$tmpPath = (string)($_FILES['video']['tmp_name'] ?? '');
if (!is_uploaded_file($tmpPath)) {
    json_out(400, ['success'=>false, 'message'=>'Fuente de archivo inválida.']);
}
$finfo = new finfo(FILEINFO_MIME_TYPE);
$mime  = $finfo->file($tmpPath);
$mimePermitidos = ['video/mp4', 'application/octet-stream']; // algunos móviles reportan octet-stream
if (!in_array($mime, $mimePermitidos, true)) {
    json_out(400, ['success'=>false, 'message'=>'El archivo no parece ser un MP4 válido.']);
}

// Directorio de destino
$dir = __DIR__ . '/videos/';
if (!is_dir($dir)) {
    if (!mkdir($dir, 0755, true)) {
        json_out(500, ['success'=>false, 'message'=>'No se pudo crear el directorio de destino.']);
    }
}
// Opcional: evitar listado de directorio en Apache
// file_put_contents($dir.'.htaccess', "Options -Indexes\n");

// Nombre seguro y único
$baseName = preg_replace('/[^a-zA-Z0-9_\-]/', '', pathinfo($nombreOriginal, PATHINFO_FILENAME));
if ($baseName === '') { $baseName = 'video'; }
$rand = bin2hex(random_bytes(8));
$nombreFinalFs = $dir . date('Ymd_His') . "_{$rand}_" . $baseName . ".mp4";

// Ruta pública/relativa para guardar en BD
$rutaRelativa = 'videos/' . basename($nombreFinalFs);

// Mover archivo
if (!move_uploaded_file($tmpPath, $nombreFinalFs)) {
    json_out(500, ['success'=>false, 'message'=>'Error al guardar el archivo en el servidor.']);
}

// Guardar en BD
if (!isset($conex) || !$conex instanceof mysqli) {
    // Si tu conexión se llama $conn, ajusta el nombre aquí y en conex.php
    json_out(500, ['success'=>false, 'message'=>'Conexión a BD no disponible.']);
}
mysqli_set_charset($conex, 'utf8mb4');

$sql = "INSERT INTO tb_video (video, ruta, activo, condicion) VALUES (?, ?, 1, 1)";
$stmt = mysqli_prepare($conex, $sql);
if (!$stmt) {
    // Limpieza del archivo si falla la preparación
    @unlink($nombreFinalFs);
    json_out(500, ['success'=>false, 'message'=>'Error al preparar la consulta.']);
}

mysqli_stmt_bind_param($stmt, 'ss', $descripcion, $rutaRelativa);
$ok = mysqli_stmt_execute($stmt);
$insertId = $ok ? mysqli_insert_id($conex) : 0;
mysqli_stmt_close($stmt);

if (!$ok) {
    @unlink($nombreFinalFs);
    json_out(500, ['success'=>false, 'message'=>'Error al guardar en la base de datos.']);
}

// Éxito
json_out(200, [
    'success'  => true,
    'message'  => 'Video agregado correctamente.',
    'id'       => $insertId,
    'ruta'     => $rutaRelativa,
    'mime'     => $mime,
    'size'     => $size
]);
