<?php
// registrar_pago_publicidad.php — con observación NUEVA/RECARGA
declare(strict_types=1);
require 'conex.php';
header('Content-Type: application/json; charset=utf-8');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  echo json_encode(['success' => false, 'message' => 'Método no permitido.']);
  exit;
}
if (!isset($conex) || !($conex instanceof mysqli)) {
  echo json_encode(['success' => false, 'message' => 'Sin conexión a la base de datos.']);
  exit;
}

@$conex->set_charset('utf8mb4');
@mysqli_query($conex, "SET time_zone='-05:00'");

/* ===== Entrada ===== */
$idPublicidad = (int)($_POST['id'] ?? 0);
$metodo       = trim((string)($_POST['metodo'] ?? ''));
$monto        = (float)($_POST['monto'] ?? 0);
$proximaFecha = trim((string)($_POST['proxima_fecha'] ?? ''));

/* ===== Validaciones ===== */
if ($idPublicidad <= 0 || $metodo === '' || $monto <= 0 || $proximaFecha === '') {
  echo json_encode(['success' => false, 'message' => 'Faltan datos obligatorios.']);
  exit;
}
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $proximaFecha)) {
  echo json_encode(['success' => false, 'message' => 'Fecha inválida (YYYY-MM-DD).']);
  exit;
}

/* ===== Resolver id_marca desde la publicidad ===== */
$sqlMarca = "SELECT id_marca FROM tb_publicidad WHERE id = ? LIMIT 1";
if (!$st = $conex->prepare($sqlMarca)) {
  echo json_encode(['success' => false, 'message' => 'Error preparando consulta de marca.']);
  exit;
}
$st->bind_param('i', $idPublicidad);
$st->execute();
$res = $st->get_result();
$row = $res ? $res->fetch_assoc() : null;
$st->close();
$idMarca = $row ? (int)$row['id_marca'] : 0;

if ($idMarca <= 0) {
  echo json_encode(['success' => false, 'message' => 'No se pudo resolver la marca asociada.']);
  exit;
}

/* ===== Determinar observación (NUEVA / RECARGA) por marca ===== */
$observacion = 'NUEVA';
$sqlCount = "SELECT COUNT(*) 
             FROM tb_pago p 
             JOIN tb_publicidad pub ON pub.id = p.id_publicidad
             WHERE pub.id_marca = ?";
if (!$st = $conex->prepare($sqlCount)) {
  echo json_encode(['success' => false, 'message' => 'Error preparando conteo de pagos.']);
  exit;
}
$st->bind_param('i', $idMarca);
$st->execute();
/* Usamos bind_result para compatibilidad */
$st->bind_result($cnt);
$st->fetch();
$st->close();
if ((int)$cnt > 0) {
  $observacion = 'RECARGA';
}

/* ===== Transacción: insertar pago + actualizar fecha_fin ===== */
$conex->begin_transaction();
try {
  // INSERT en tb_pago (añadimos la columna observacion)
  $sqlIns = "INSERT INTO tb_pago
               (id_publicidad, tipo_pago, valor, fecha, fecha_corte, observacion, condicion)
             VALUES
               (?, ?, ?, NOW(), ?, ?, 1)";
  if (!$st = $conex->prepare($sqlIns)) {
    throw new Exception('Error preparando INSERT en tb_pago: ' . $conex->error);
  }
  // tipos: i (id) s (texto) d (decimal) s (fecha) s (observacion)
  $obs = ($observacion === '' || $observacion === null) ? 'NUEVA' : $observacion;
  $st->bind_param('isdss', $idPublicidad, $metodo, $monto, $proximaFecha, $obs);
  if (!$st->execute()) {
    throw new Exception('Error ejecutando INSERT en tb_pago: ' . $st->error);
  }
  $idPago = (int)$conex->insert_id;
  $st->close();

  // UPDATE fecha_fin en tb_publicidad
  $sqlUpd = "UPDATE tb_publicidad SET fecha_fin = ? WHERE id = ?";
  if (!$st2 = $conex->prepare($sqlUpd)) {
    throw new Exception('Error preparando UPDATE en tb_publicidad: ' . $conex->error);
  }
  $st2->bind_param('si', $proximaFecha, $idPublicidad);
  if (!$st2->execute()) {
    throw new Exception('Error ejecutando UPDATE en tb_publicidad: ' . $st2->error);
  }
  $st2->close();

  $conex->commit();
  echo json_encode([
    'success'      => true,
    'id_pago'      => $idPago,
    'observacion'  => $obs
  ]);
} catch (Throwable $e) {
  $conex->rollback();
  echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
