<?php
// reporte_pagos.php — URBANVIEW · Reporte de pagos por marca (Nuevos vs Recargas + faltantes)
// Esquema real usado:
// tb_pago(id, id_publicidad, tipo_pago, valor, fecha, fecha_corte, condicion, observacion)
// tb_marca(id, nombre_marca, ..., plan, valor, tiempo, condicion)
// tb_publicidad(id, fecha_inicio, fecha_fin, activo, id_marca, id_zona, id_canton, id_video, condicion)

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

if (!isset($_SESSION['usuario']) || empty($_SESSION['usuario'])) {
  header('Location: ../../index.php'); exit();
}
$usuario = $_SESSION['usuario'] ?? '';

// ===== Conexión BD (soporta $conn ó $conex)
$pathConex = realpath(__DIR__ . '/conex.php') ?: realpath(__DIR__ . '/../conex.php') ?: realpath(__DIR__ . '/../../conex.php');
if (!$pathConex) { http_response_code(500); echo 'No se encontró conex.php'; exit; }
require $pathConex;

$DB = null;
if (isset($conn) && $conn instanceof mysqli) { $DB = $conn; }
elseif (isset($conex) && $conex instanceof mysqli) { $DB = $conex; }
if (!$DB) { http_response_code(500); echo 'Sin conexión a la base de datos'; exit; }
mysqli_set_charset($DB, 'utf8mb4');
@mysqli_query($DB, "SET time_zone = '-05:00'");

// ===== Fechas por defecto (mes actual)
$hoy = new DateTimeImmutable('now', new DateTimeZone('America/Guayaquil'));
$inicioDefault = $hoy->modify('first day of this month')->format('Y-m-d');
$finDefault    = $hoy->modify('last day of this month')->format('Y-m-d');

// ===== API
$action = $_GET['action'] ?? '';

// Silenciar warnings en endpoints JSON (evita “Invalid JSON response”)
if ($action === 'list' || $action === 'detalle') {
  ini_set('display_errors', '0');
  ini_set('log_errors', '1');
  ini_set('error_log', __DIR__ . '/reporte_pagos_error.log');
}

/* -------- action=list  (agregado por marca) -------- */
if ($action === 'list') {
  $desde = $_GET['desde'] ?? $inicioDefault;
  $hasta = $_GET['hasta'] ?? $finDefault;

  $desde = preg_match('/^\d{4}-\d{2}-\d{2}$/', $desde) ? $desde : $inicioDefault;
  $hasta = preg_match('/^\d{4}-\d{2}-\d{2}$/', $hasta) ? $hasta : $finDefault;

  // 1) Contrato por marca: meses_contrato (usa m.tiempo si >0; si no, suma de meses por tb_publicidad)
  $sqlContratos = "
    SELECT
      m.id                                AS id_marca,
      m.nombre_marca,
      COALESCE(m.plan,'')                 AS plan,
      COALESCE(m.valor,0)                 AS valor_mensual,
      COALESCE(m.tiempo,0)                AS tiempo_marca,
      MIN(p.fecha_inicio)                 AS fecha_inicio_min,
      MAX(p.fecha_fin)                    AS fecha_fin_max,
      SUM(
        CASE
          WHEN p.fecha_inicio IS NULL OR p.fecha_fin IS NULL THEN 0
          ELSE PERIOD_DIFF(DATE_FORMAT(p.fecha_fin, '%Y%m'), DATE_FORMAT(p.fecha_inicio, '%Y%m')) + 1
        END
      )                                   AS meses_publicidad
    FROM tb_marca m
    LEFT JOIN tb_publicidad p ON p.id_marca = m.id AND p.condicion = 1
    WHERE m.condicion = 1
    GROUP BY m.id, m.nombre_marca, m.plan, m.valor, m.tiempo
  ";
  $rsC = $DB->query($sqlContratos);
  if (!$rsC) { header('Content-Type: application/json; charset=utf-8'); echo json_encode(['success'=>false,'error'=>'Contratos: '.mysqli_error($DB)], JSON_UNESCAPED_UNICODE); exit; }
  $contratos = [];
  while ($r = $rsC->fetch_assoc()) { $contratos[(int)$r['id_marca']] = $r; }

  // 2) Pagos en RANGO por marca (para Nuevos/Recargas/Total)
  $sqlPagosRango = "
    SELECT
      m.id AS id_marca,
      SUM(CASE WHEN g.observacion='NUEVA'   THEN g.valor ELSE 0 END) AS valor_nuevos,
      SUM(CASE WHEN g.observacion='RECARGA' THEN g.valor ELSE 0 END) AS valor_recargas,
      SUM(g.valor)                                                AS valor_total,
      SUM(CASE WHEN g.observacion='NUEVA'   THEN 1 ELSE 0 END)      AS cant_nuevos,
      SUM(CASE WHEN g.observacion='RECARGA' THEN 1 ELSE 0 END)      AS cant_recargas,
      COUNT(*)                                                    AS cant_pagos
    FROM tb_pago g
    INNER JOIN tb_publicidad p ON p.id = g.id_publicidad
    INNER JOIN tb_marca m ON m.id = p.id_marca
    WHERE g.condicion = 1
      AND g.fecha BETWEEN ? AND ?
    GROUP BY m.id
  ";
  $stmtPR = $DB->prepare($sqlPagosRango);
  if (!$stmtPR) { header('Content-Type: application/json; charset=utf-8'); echo json_encode(['success'=>false,'error'=>'Pagos rango: '.mysqli_error($DB)], JSON_UNESCAPED_UNICODE); exit; }
  $stmtPR->bind_param('ss', $desde, $hasta);
  $stmtPR->execute();
  $resPR = $stmtPR->get_result();
  $pagosRango = [];
  while ($r = $resPR->fetch_assoc()) { $pagosRango[(int)$r['id_marca']] = $r; }
  $stmtPR->close();

  // 3) Meses pagados (acumulado) y última fecha_corte por marca (SIN filtrar por rango)
  $sqlMesesPagados = "
    SELECT
      m.id AS id_marca,
      COUNT(DISTINCT DATE_FORMAT(g.fecha, '%Y-%m')) AS meses_pagados,
      MAX(g.fecha_corte)                            AS ultima_fecha_corte
    FROM tb_pago g
    INNER JOIN tb_publicidad p ON p.id = g.id_publicidad
    INNER JOIN tb_marca m ON m.id = p.id_marca
    WHERE g.condicion = 1
    GROUP BY m.id
  ";
  $rsMP = $DB->query($sqlMesesPagados);
  if (!$rsMP) { header('Content-Type: application/json; charset=utf-8'); echo json_encode(['success'=>false,'error'=>'Meses pagados: '.mysqli_error($DB)], JSON_UNESCAPED_UNICODE); exit; }
  $mesesPag = [];
  while ($r = $rsMP->fetch_assoc()) { $mesesPag[(int)$r['id_marca']] = $r; }

  // 4) Fusión y cálculo final
  $ids = array_unique(array_merge(array_keys($contratos), array_keys($pagosRango), array_keys($mesesPag)));
  $data = [];
  foreach ($ids as $id_marca) {
    $c = $contratos[$id_marca] ?? [
      'id_marca'=>$id_marca,'nombre_marca'=>'(sin datos)','plan'=>'','valor_mensual'=>0,'tiempo_marca'=>0,
      'fecha_inicio_min'=>null,'fecha_fin_max'=>null,'meses_publicidad'=>0
    ];
    $p = $pagosRango[$id_marca] ?? [
      'valor_nuevos'=>0,'valor_recargas'=>0,'valor_total'=>0,'cant_nuevos'=>0,'cant_recargas'=>0,'cant_pagos'=>0
    ];
    $mp = $mesesPag[$id_marca] ?? ['meses_pagados'=>0,'ultima_fecha_corte'=>null];

    $valor_mensual = (float)$c['valor_mensual'];
    $tiempo_marca  = (int)$c['tiempo_marca'];
    $meses_publi   = (int)$c['meses_publicidad'];

    $meses_contrato = $tiempo_marca > 0 ? $tiempo_marca : $meses_publi;
    $meses_pagados  = (int)$mp['meses_pagados'];

    $meses_faltantes = max(0, $meses_contrato - $meses_pagados);
    $valor_faltante  = $meses_faltantes * $valor_mensual;

    $data[] = [
      'id_marca'           => (int)$id_marca,
      'nombre_marca'       => (string)$c['nombre_marca'],
      'plan'               => (string)$c['plan'],
      'valor_mensual'      => round($valor_mensual, 2),
      'fecha_inicio'       => $c['fecha_inicio_min'] ? (string)$c['fecha_inicio_min'] : '',
      'fecha_fin'          => $c['fecha_fin_max'] ? (string)$c['fecha_fin_max'] : '',
      'meses_contrato'     => (int)$meses_contrato,
      'meses_pagados'      => (int)$meses_pagados,
      'meses_faltantes'    => (int)$meses_faltantes,
      'valor_faltante'     => round($valor_faltante, 2),

      'cant_nuevos'        => (int)$p['cant_nuevos'],
      'cant_recargas'      => (int)$p['cant_recargas'],
      'valor_nuevos'       => round((float)$p['valor_nuevos'], 2),
      'valor_recargas'     => round((float)$p['valor_recargas'], 2),
      'valor_total'        => round((float)$p['valor_total'], 2),

      'ultima_fecha_corte' => $mp['ultima_fecha_corte'] ? (string)$mp['ultima_fecha_corte'] : ''
    ];
  }

  header('Content-Type: application/json; charset=utf-8');
  echo json_encode(['success'=>true, 'data'=>$data], JSON_UNESCAPED_UNICODE);
  exit;
}

/* -------- action=detalle  (pagos por marca en el rango) -------- */
if ($action === 'detalle') {
  $id_marca = (int)($_GET['id_marca'] ?? 0);
  $desde = $_GET['desde'] ?? $inicioDefault;
  $hasta = $_GET['hasta'] ?? $finDefault;
  $desde = preg_match('/^\d{4}-\d{2}-\d{2}$/', $desde) ? $desde : $inicioDefault;
  $hasta = preg_match('/^\d{4}-\d{2}-\d{2}$/', $hasta) ? $hasta : $finDefault;

  $sql = "
    SELECT
      m.nombre_marca,
      g.id, g.id_publicidad,
      g.tipo_pago,
      g.fecha,
      g.valor,
      COALESCE(g.fecha_corte,'') AS fecha_corte,
      COALESCE(g.observacion,'') AS observacion
    FROM tb_pago g
    INNER JOIN tb_publicidad p ON p.id = g.id_publicidad
    INNER JOIN tb_marca m ON m.id = p.id_marca
    WHERE m.id = ?
      AND g.condicion = 1
      AND g.fecha BETWEEN ? AND ?
    ORDER BY g.fecha DESC, g.id DESC
  ";
  $stmt = $DB->prepare($sql);
  if (!$stmt) { header('Content-Type: application/json; charset=utf-8'); echo json_encode(['success'=>false,'error'=>'Detalle: '.mysqli_error($DB)], JSON_UNESCAPED_UNICODE); exit; }
  $stmt->bind_param('iss', $id_marca, $desde, $hasta);
  $stmt->execute();
  $rs = $stmt->get_result();
  $rows = [];
  $marca = '';
  while ($r = $rs->fetch_assoc()) { $rows[] = $r; $marca = $r['nombre_marca']; }
  $stmt->close();

  header('Content-Type: application/json; charset=utf-8');
  echo json_encode(['success'=>true, 'marca'=>$marca, 'rows'=>$rows], JSON_UNESCAPED_UNICODE);
  exit;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="UTF-8" />
<title>URBANVIEW · Reporte de pagos por marca</title>
<link rel="shortcut icon" href="../../resources/img/logour.ico" />
<meta name="viewport" content="width=device-width, initial-scale=1" />

<!-- Bootstrap 5 -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
<!-- FontAwesome -->
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css" rel="stylesheet" />
<!-- DataTables -->
<link href="https://cdn.datatables.net/1.13.4/css/dataTables.bootstrap5.min.css" rel="stylesheet" />
<!-- DataTables Buttons -->
<link href="https://cdn.datatables.net/buttons/2.4.2/css/buttons.bootstrap5.min.css" rel="stylesheet" />

<style>
:root{ --topbar-h: 56px; }
html, body { height:100%; margin:0; overflow:hidden; font-family: Arial, Helvetica, sans-serif; }
body { display:flex; flex-direction:column; }
.topbar{ height:var(--topbar-h); background:#343a40; color:#fff; display:flex; align-items:center; justify-content:space-between; padding:0 12px; position:fixed; top:0; left:0; right:0; z-index:1000; }
.wrapper{ display:flex; flex-grow:1; margin-top:var(--topbar-h); height:calc(100vh - var(--topbar-h)); overflow:hidden; }
.sidebar{ width:240px; background:#343a40; color:#fff; padding:12px; overflow-y:auto; flex-shrink:0; transition:width .25s ease; }
.sidebar a{ color:#fff; text-decoration:none; padding:8px 10px; display:flex; align-items:center; border-radius:6px; }
.sidebar a:hover{ background:#495057; }
.sidebar.collapsed{ width:76px; }
.sidebar.collapsed .text-label{ display:none; }
.main-content{ flex-grow:1; height:100%; display:flex; flex-direction:column; margin-left:240px; padding:12px 16px; transition:margin-left .25s ease; overflow:auto; }
.sidebar.collapsed ~ .main-content{ margin-left:76px; }

@media (max-width:768px){
  .sidebar{ position:fixed; top:var(--topbar-h); left:0; height:calc(100% - var(--topbar-h)); z-index:999; transform:translateX(-100%); }
  .sidebar.show{ transform:translateX(0); }
  .main-content{ margin-left:0 !important; }
}

.table-sm td, .table-sm th { padding:.35rem .5rem; }
.badge.rounded-pill{ font-weight:500; }
</style>
</head>
<body>
<header class="topbar">
  <button class="btn btn-dark btn-sm" onclick="toggleSidebar()" aria-label="Abrir/cerrar menú">
    <i class="fas fa-bars"></i>
  </button>
  <span>URBANVIEW · Reporte de Pagos</span>
  <div class="d-flex align-items-center gap-2">
    <span><?=htmlspecialchars($usuario, ENT_QUOTES, 'UTF-8');?></span>
    <span id="fechaActual" class="d-none d-md-inline"></span>
    <span id="horaActual"  class="d-none d-md-inline"></span>
    <button class="btn btn-sm btn-danger" onclick="cerrarSesion(event)" title="Cerrar sesión">
      <i class="fas fa-sign-out-alt"></i>
    </button>
  </div>
</header>

<div class="wrapper">
  <nav class="sidebar"><?php include 'menu.php'; ?></nav>
  <div class="main-content">

    <!-- Filtros -->
    <div class="card shadow-sm mb-2">
      <div class="card-body py-2">
        <div class="row g-2 align-items-end">
          <div class="col-6 col-md-3">
            <label class="form-label mb-1">Desde</label>
            <input type="date" id="fDesde" class="form-control form-control-sm" value="<?=$inicioDefault?>" />
          </div>
          <div class="col-6 col-md-3">
            <label class="form-label mb-1">Hasta</label>
            <input type="date" id="fHasta" class="form-control form-control-sm" value="<?=$finDefault?>" />
          </div>
          <div class="col-12 col-md-auto">
            <button id="btnFiltrar" class="btn btn-primary btn-sm">
              <i class="fas fa-filter me-1"></i>Filtrar
            </button>
            <button id="btnMes" class="btn btn-outline-secondary btn-sm">
              <i class="fas fa-calendar-day me-1"></i>Mes actual
            </button>
          </div>
        </div>
      </div>
    </div>

    <!-- Tabla -->
    <div class="card shadow-sm">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <h6 class="mb-0">Pagos por marca (Nuevos / Recargas) y faltantes</h6>
          <div id="totales" class="small text-muted"></div>
        </div>
        <div class="table-responsive">
          <table id="tablaReporte" class="table table-sm table-striped table-hover align-middle" style="width:100%">
            <thead>
              <tr>
                <th>Marca</th>
                <th>Plan</th>
                <th>$ Mensual</th>
                <th>Contrato<br>(meses)</th>
                <th>Pagados<br>(meses)</th>
                <th>Faltan<br>(meses)</th>
                <th>Nuevos<br>(# / $)</th>
                <th>Recargas<br>(# / $)</th>
                <th>Valor total</th>
                <th>Valor faltante</th>
                <th>Rango contrato</th>
                <th>Últ. fecha corte</th>
                <th>Detalle</th>
              </tr>
            </thead>
            <tbody></tbody>
            <tfoot>
              <tr class="fw-semibold">
                <td colspan="6" class="text-end">Totales:</td>
                <td id="tf_nuevos"></td>
                <td id="tf_recargas"></td>
                <td id="tf_total"></td>
                <td id="tf_faltante"></td>
                <td colspan="3"></td>
              </tr>
            </tfoot>
          </table>
        </div>
      </div>
    </div>

    <div class="text-center text-muted small mt-2">© <?=date('Y');?> URBANVIEW</div>
  </div>
</div>

<!-- Modal detalle -->
<div class="modal fade" id="modalDetalle" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header py-2">
        <h6 class="modal-title">Detalle de pagos — <span id="md_marca"></span></h6>
        <button class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body p-2">
        <div class="table-responsive">
          <table class="table table-sm table-striped align-middle">
            <thead>
              <tr>
                <th>Fecha</th>
                <th>Tipo</th>
                <th>Valor</th>
                <th>Fecha corte</th>                
                <th>Obs.</th>
              </tr>
            </thead>
            <tbody id="md_tbody"></tbody>
          </table>
        </div>
      </div>
      <div class="modal-footer py-2">
        <button class="btn btn-secondary btn-sm" data-bs-dismiss="modal">Cerrar</button>
      </div>
    </div>
  </div>
</div>

<!-- JS -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/dataTables.buttons.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.bootstrap5.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.2.7/pdfmake.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.2.7/vfs_fonts.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.html5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.print.min.js"></script>

<script>
// Utilidades
function notify(type,title,text){ alert((title?title+': ':'')+(text||'')); }
function formatoMon(x){ return (Number(x)||0).toLocaleString('es-EC',{minimumFractionDigits:2, maximumFractionDigits:2}); }

// Reloj
function actualizarHora(){
  const tz='America/Guayaquil';
  const ahora=new Date();
  document.getElementById('fechaActual').textContent=
    ahora.toLocaleDateString('es-EC',{weekday:'long',year:'numeric',month:'long',day:'numeric', timeZone:tz});
  document.getElementById('horaActual').textContent=
    ahora.toLocaleTimeString('es-EC',{hour:'2-digit',minute:'2-digit',second:'2-digit',hour12:false, timeZone:tz});
}
setInterval(actualizarHora,1000); actualizarHora();

// Sidebar
const sidebar=document.querySelector('.sidebar');
function toggleSidebar(){
  if (!sidebar) return;
  if (window.innerWidth<=768) sidebar.classList.toggle('show');
  else sidebar.classList.toggle('collapsed');
}
document.addEventListener('click',(e)=>{
  if (window.innerWidth>768) return;
  const btn = e.target.closest('button[onclick="toggleSidebar()"]');
  if (!btn && sidebar && !sidebar.contains(e.target)) sidebar.classList.remove('show');
});

// Cerrar sesión
function cerrarSesion(e){
  if (e) e.preventDefault();
  if (!confirm('¿Cerrar sesión?')) return;
  fetch('../../controllers/usuario.php',{ method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body:new URLSearchParams({boton:'cerrar'})})
  .then(r=>{ if (r.ok) location.href='../../index.php'; else throw new Error('Error al cerrar sesión'); })
  .catch(err=>notify('error','Error',err.message));
}

// DataTable
let dt;
function cargar(){
  const desde = document.getElementById('fDesde').value;
  const hasta = document.getElementById('fHasta').value;

  if (dt) { dt.destroy(); $('#tablaReporte tbody').empty(); }

  dt = $('#tablaReporte').DataTable({
    ajax: {
      url: 'reporte_pagos.php',
      data: { action:'list', desde, hasta },
      error: function(xhr){
        console.error('Respuesta del servidor:', xhr.responseText);
        alert('Error al cargar. Revisa la consola del navegador.');
      },
      dataSrc: function(json){
        if (!json || json.success === false) {
          const msg = (json && json.error) ? json.error : 'Fallo al cargar';
          alert(msg); return [];
        }
        // Totales footer
        let t_n = 0, t_n_$ = 0, t_r = 0, t_r_$ = 0, t_total = 0, t_faltante = 0;
        json.data.forEach(r=>{
          t_n   += Number(r.cant_nuevos||0);
          t_n_$ += Number(r.valor_nuevos||0);
          t_r   += Number(r.cant_recargas||0);
          t_r_$ += Number(r.valor_recargas||0);
          t_total    += Number(r.valor_total||0);
          t_faltante += Number(r.valor_faltante||0);
        });
        document.getElementById('tf_nuevos').textContent   = `${t_n} / $ ${formatoMon(t_n_$)}`;
        document.getElementById('tf_recargas').textContent = `${t_r} / $ ${formatoMon(t_r_$)}`;
        document.getElementById('tf_total').textContent    = `$ ${formatoMon(t_total)}`;
        document.getElementById('tf_faltante').textContent = `$ ${formatoMon(t_faltante)}`;
        document.getElementById('totales').textContent     = `Marcas: ${json.data.length}`;
        return json.data;
      }
    },
    columns: [
      { data:'nombre_marca' },
      { data:'plan' },
      { data:'valor_mensual', render:d=>`$ ${formatoMon(d)}`, className:'text-end' },
      { data:'meses_contrato', className:'text-center' },
      { data:'meses_pagados', className:'text-center' },
      { data:'meses_faltantes', className:'text-center' },
      { data:null, render:r=>`${r.cant_nuevos} / $ ${formatoMon(r.valor_nuevos)}`, className:'text-end' },
      { data:null, render:r=>`${r.cant_recargas} / $ ${formatoMon(r.valor_recargas)}`, className:'text-end' },
      { data:'valor_total', render:d=>`$ ${formatoMon(d)}`, className:'text-end' },
      { data:'valor_faltante', render:d=>`$ ${formatoMon(d)}`, className:'text-end' },
      { data:null, render:r=>{
          const f1 = r.fecha_inicio ? new Date(r.fecha_inicio).toLocaleDateString('es-EC') : '-';
          const f2 = r.fecha_fin    ? new Date(r.fecha_fin   ).toLocaleDateString('es-EC') : '-';
          return `${f1} → ${f2}`;
        }},
      { data:'ultima_fecha_corte', render:d=> d ? new Date(d).toLocaleDateString('es-EC') : '-', className:'text-center' },
      { data:null, orderable:false, render:r=>`<button class="btn btn-sm btn-outline-primary" onclick="verDetalle(${r.id_marca})"><i class="fa fa-eye me-1"></i>Ver</button>` }
    ],
    order: [[0,'asc']],
    pageLength: 10,
    lengthMenu: [10,25,50,100],
    dom: 'Bfrtip',
    buttons: [
      { extend:'excelHtml5', title:()=>`Reporte_Pagos_${desde}_a_${hasta}` },
      { extend:'pdfHtml5',   title:()=>`Reporte_Pagos_${desde}_a_${hasta}`, orientation:'landscape', pageSize:'A4' },
      { extend:'print',      title:'' }
    ],
    language: { url: 'https://cdn.datatables.net/plug-ins/1.13.4/i18n/es-ES.json' }
  });
}

function verDetalle(id_marca){
  const desde = document.getElementById('fDesde').value;
  const hasta = document.getElementById('fHasta').value;
  fetch(`reporte_pagos.php?action=detalle&id_marca=${id_marca}&desde=${desde}&hasta=${hasta}`)
  .then(r=>r.json())
  .then(j=>{
    if (!j.success){ alert(j.error || 'No se pudo obtener detalle'); return; }
    document.getElementById('md_marca').textContent = j.marca || '';
    const tb = document.getElementById('md_tbody');
    tb.innerHTML = '';
    j.rows.forEach(x=>{
      const tr = document.createElement('tr');
      const fecha = x.fecha ? new Date(x.fecha).toLocaleDateString('es-EC') : '-';
      const corte = x.fecha_corte ? new Date(x.fecha_corte).toLocaleDateString('es-EC') : '-';
      const badge = x.observacion === 'NUEVA'
        ? '<span class="badge rounded-pill text-bg-info">Nuevo</span>'
        : '<span class="badge rounded-pill text-bg-success">Recarga</span>';
      tr.innerHTML = `
        <td>${fecha}</td>
        <td>${badge}</td>
        <td class="text-end">$ ${formatoMon(x.valor)}</td>
        <td>${corte}</td>        
        <td>${x.observacion ? x.observacion.replace(/</g,'&lt;') : ''}</td>
      `;
      tb.appendChild(tr);
    });
    new bootstrap.Modal(document.getElementById('modalDetalle')).show();
  })
  .catch(err=>alert(err.message));
}

// Filtros
document.getElementById('btnFiltrar').addEventListener('click', cargar);
document.getElementById('btnMes').addEventListener('click', ()=>{
  const d = new Date(); d.setDate(1);
  const y=d.getFullYear(), m=String(d.getMonth()+1).padStart(2,'0');
  const ultimo = new Date(y, d.getMonth()+1, 0).getDate();
  document.getElementById('fDesde').value = `${y}-${m}-01`;
  document.getElementById('fHasta').value = `${y}-${m}-${String(ultimo).padStart(2,'0')}`;
  cargar();
});

// Inicial
document.addEventListener('DOMContentLoaded', cargar);
</script>
</body>
</html>
